package Parse::Prep;

# prep - an unusual preprocessor
#
# Copyright 1994-2000 Carlo von Loesch (lynx@prep.pages.de).
# Licensed to you according to the same legalese as perl itself.

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
#se Carp;

$VERSION = '4.2';

# so for once this is eurocentristic, add the language you miss
# this will obviously never be able to handle all of the world's
# languages - but it was never meant to. single character language
# flags is good for the "I)parlo italiano" syntax.
#
my %l = (
	de => 'D',
	en => 'E',
	fr => 'F',
	it => 'I',
	jp => 'J',
	es => 'S',
);

# Preloaded methods go here.
# Autoload methods go after =cut, and are processed by the autosplit program.

sub new {
    my ($class, %v) = @_;

    $v{Pkg} ||= caller;
    bless \%v, $class;
}

# one-shot parsing calls
sub prepfile {
    my $me = shift;
    my $file = shift;
    my $flags = shift;

    $me->init($file, $flags);
    $me->parsefile($file);
    $me->done;
}
sub prep {
    my $me = shift;
    my $rawdata = shift;
    my $flags = shift;

    $me->init(undef, $flags);
    $me->parse($rawdata);
    $me->done;
}

# fragmented parsing calls
sub init {
    # init's arguments should be replaced by a hash like new()'s
    my $me = shift;
    my $file = shift;
    my $flags = shift;
    my $lang = shift;

    $me->{Flags} = $flags if defined $flags;
    $me->{OR} = $me->flag('|') ? 1 : 0;
    $me->{Language} = $lang if $lang;

    if (lc $me->{Language}) {
	if ( $l{$me->{Language}} ) {
	    $me->define( $l{$me->{Language}} );
	} else {
	    $me->define( $me->{Language} );
	}
    } elsif ($file && $file =~ /\.(\w\w)\./) {
	$me->define( $l{$1} ) if $l{$1};
    }
    $me->{block} = 0;
    $me->{level} = 0;
    $me->interpret('init');
}
sub done {
    my $me = shift;
    $me->interpret('done');
}
sub parsefile {
    my $me = shift;
    my $x = shift;

    local(*I);
    open(I, $x) or die "unable to open $x (current dir: ". `pwd` .')';
    # not exactly efficient, but this can be improved later
    $me->parseline($_) while <I>;
    close I;
}
sub parse {
    my $me = shift;
    my $rawdata = shift;

    # not exactly efficient, but this can be improved later
    foreach(split(/\n/, $rawdata)) {
	$me->parseline($_);
    }
}
# unusual inspector, used by LYM
sub encountered {
    my $me = shift;
    my $x = shift;
    return $me->{FlagsFound}{$x};
}


## END OF PUBLIC METHODS


# here comes the main engine
sub parseline {
    my $me = shift;
    $_ = shift;

    my $nl;
    s/\r?\n?$//;	# remove line delimiters
    if (/(.*)\\$/o) {	# support for \ line merging
	    $nl = '';
	    $_ = $1;
    } else {
	    $nl = "\n";
    }

    study;

    # support for flag) syntax
    my ($pre, $post) = /^(\w+)\)(.*)/o;
    if ($pre) {
	my $state = 0;
	foreach(split('', $pre)) {
		$me->{FlagsFound}{$pre}++;
		if ($me->{OR} ^ (index($me->{Flags}, $_) < 0)) {
		    $state = 1; last;
		}
	}
	return if $state ^ $me->{OR};
	$_ = $post;
    }

    # support for leading #
    if (/^#/) {
	# else/endif need to be checked before ifdef
	$me->else($1), return if /^#else\b(.*)/o;
	$me->endif($2), return if /^#(endif|fi)\b(.*)/o;
	$me->ifdef($1), return if /^#if\s+(.+)\s*$/o;
	$me->ifndef($1), return if /^#ifn\s+(.+)\s*$/o;

	unless ($me->{block}) {
	    $me->define($2), return if /^#de(f|fine)\s+(\S+)\s*$/o;
	    $me->prepfile($2), return if /^#in(c|clude)\s+(\S+)\s*$/o;
#	    $me->require($2), return if /^#(require|doctype)\s+(\S+)\s*$/o;
#	    $me->pragma($1), return if /^#pragma\s+(\S+)\s*$/o;
	    $me->interpret($1, $2, 'do'), return if /^#(\w+)\s*(.*?)\s*$/o;
	}
	return;
    }

    # support for inline ((( )))
    #
    s/\(\(\(\s*(\w+)\s*(.*?)\s*\)\)\)/ $me->interpret($1,$2,'m') /ge;
    #
    # this should probably operate on a whole file at once, not line by line
    # maybe after line-by-line processing
    #
    # the whole "prepline" approach is a bit historic, a more efficient
    # technique could be implemented. but hey, it does the job well, too.

    unless ($me->{block}) {
	if ($me->{CatchText}) {
	    $_ = $me->interpret('text', $_.$nl);
	}
	print $_, $nl if defined $_;
    }
}

sub interpret {
	my $me = shift;
	my $cmd = shift;
	my $args = shift;
	my $prefix = shift;

	return ${$me}{$cmd} if ${$me}{$cmd} &&! $args;

	my $func = $me->{Pkg}. ($prefix ? "::$prefix\_" : '::') .$cmd;
	unless (defined &$func) {
	    die("broken instruction: #$cmd $args ($prefix\_$cmd)") if $prefix;
	    return undef;
	}
	# if $func were in an object we could use $obj->can($func)

	# I could use an easier syntax here
	my @A = $prefix ? split(/\s+\,\s+/, $args) : ($args);

	my $ret;
	$@ = undef;
	eval "\$ret = &$func(\@A)";
	die "\n#$cmd $args:\n", $@, "\n" if $@;
	return $ret;
}

sub define {
	my $me = shift;
	my $x = shift;
	$me->{Flags} .= $x;
}
sub flag {
	my $me = shift;
	my $x = shift;
	index($me->{Flags}, $x) >= 0;
}

# ifdef/endif logic starts here
#
sub IFDEB { 1 }		# inline macro flag for debugging activation
sub expr {
	my $me = shift;
	$_ = shift;

	my $true;
	if (/^!\s*(\w+)$/) {
		$true = ! $me->flag($1);
	} elsif (/^(\w+)$/) {
		$true = $me->flag($1);
        print "- x-expr $1 is $true -\n\n" if IFDEB;
#	} else {
#		eval '$true='. $_ .';';
	}

        print "- expr $_ is $true -\n\n" if IFDEB;
	return $true;
}
sub ifdef {
	my $me = shift;
	my $x = shift;

	$me->{FlagsFound}{$x}++;
	$me->{level}++;
        unless ($me->{block}) {
		$me->{block} = $me->{level} unless $me->expr($x);
        }
        print "- $x ? - level $me->{level} - block $me->{block}\n\n" if IFDEB;
}
sub ifndef {
	my $me = shift;
	my $x = shift;

	$me->{FlagsFound}{$x}++;
	$me->{level}++;
        unless ($me->{block}) {
		$me->{block} = $me->{level} if $me->expr($x);
        }
        print "- $x ? - level $me->{level} - block $me->{block}\n\n" if IFDEB;
}
sub else {
	my $me = shift;

	print "- else -\n\n" if IFDEB;
	$me->{block} = $me->{level}, return unless $me->{block};
	$me->{block}=0 if $me->{level} == $me->{block};
}
sub endif {
	my $me = shift;

	$me->{block}=0 if $me->{level} == $me->{block};
	$me->{level}--;
	die 'endif without if' if $me->{level} < 0;
	print "- endif - level $me->{level} - block $me->{block}\n\n" if IFDEB;
}

1;
__END__


sub pragma {
	local($_) = @_;
	$OR = 0 if /\bAND\b/io;
	$OR = 1 if /\bOR\b/io;
}
sub require {
	local($x) = @_;
	$x .= '.pl';
	$t = './'.$x;
	require $t, return if -r $t;
	# $t = $HOME.'/'.$x.'.pl';
	# require $t, return if -r $t;
	# $t = $LIB.'/'.$x.'.pl';
	require $x;
}

1;

__END__

=head1 NAME

Parse::Prep - An unusual preprocessor

=head1 SYNOPSIS

  use Parse::Prep;
  $prep = new Parse::Prep ( Language => 'fr', Flags => 'n' );

=head1 DESCRIPTION

Why yet another preprocessor? Well - embedded-perl-approaches to making
webpages more intelligent assume the user can deal with real perl. 
This little meta-html-language doesn't do that, it doesn't even
replicate structures like CIPP does. all it gives you is a syntax to
define macros in perl that the htmlist can use, plus a syntax to handle
multi-language and multi-browser sites. And it does so in less than
500 lines of perl.

Inline macros have an unusal syntax: (((my macro))). The advantages
of this syntax - it sticks out in a crowd of text and markup.

prep parses "X)" flag prefixes for each line of input, if you set a
language like "it", all lines starting with "I)" will be passed to
the output while lines starting with "D)" won't.

prep knows XY) combined flag prefixes.
prep knows "(((pragma or)))" to apply OR instead of AND with
combined-flag-prefixes.

prep supports a cpp-style syntax, which is usually not suited to the
end users, so don't tell them about it.

prep knows hierarchical #if/#else/#endif constructs.
prep knows #include and #ifn (if not).
prep removes all unknown lines starting with #, like comments.

prep knows #require to extend the # command set (#doctype is also accepted),
those commands can also be called via the (((require ...))) syntax.

=head1 AUTHOR

Carlo von Loesch <lynx@prep.pages.de> http://perl.pages.de/

=head1 HISTORY

The prep.pl parser library was written in 1994 and served in
website and system administration ever since. This is the
perl5 rewrite. After looking into SGML and XML I realized that for many
purposes my own prep syntax was a better option.

=head1 ANECDOTE

There's a little funny story about the multi-language capabilities of
prep that I may add one day.

=head1 SEE ALSO

perl(1).

=cut
